/*

	Copyright (c) 2004 PXI Project Team

	Permission is hereby granted, free of charge, to any person obtaining a copy of
	this software and associated documentation files (the "Software"), to deal in the
	Software without restriction, including without limitation the rights to use,
	copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
	Software, and to permit persons to whom the Software is furnished to do so,
	subject to the following conditions:

	The above copyright notice and this permission notice shall be included in all copies
	or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
	INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
	PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
	HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
	OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

/*

	CLASS variant
 
	The variant class is used to hold a value that may be represented by multiple data types.
	Its job is to store the value internally, and convert between the various types as necessary.
	Obviously, you want this to be as quick as possible, and the functions tend to be fairly
	short, so it tends to do a lot of inlining.
 
	RETROFIT: We'll be using variants to store nonscalar values (such as objects) because the
	architecture is already equipped to handle variants, meaning we don't have to modify anything
	to be able to push our custom values around. It is critical that we not attempt to use such
	'packed' variants as scalar values, or we will most likely be rewarded with a system failure.
 
	TYPICAL USAGE
 
	Generally, you create variant with one of its two primary data types - a string or a double.
	You just use normal assignment to add a value, and let the object figure out how to store it.
	If you want a number out of it, you call intVal(), if you want a string out of it, you call
	strVal(). Note that it will blow your program to kingdom come if it can't make the conversion.
 
	VVR AND ACR
 
	The latest version of variant supports a new feature called "referencing," which is basically
	a fancy way of storing a pointer to a variant. The variant encapsulates the pointer mechanism,
	and automatically routes calls appropriately.
 
	The first type of referencing is called Variant to Variant Referencing, or VVR. This is basically
	the mechanism just described. If you want to create a reference variant, you just say something
	like: variant vRef; vRef.linkTo( vTarg ); If you want to turn a variant back into itself, use
	vRef.unLink();. If you want to make a variant into a copy of its target, use vRef.flatten();
 
	Another type of referencing is called Automatic Chunk Referencing (ACR). This works in tandem
	with VVR, or by itself. Basically, you can define a certain chunk type and range, and the variant
	will automatically adjust its calls to work within those constraints. What makes this feature
	unique is that it operates independently of the VVR system, so you can have a variant that is
	a reference to a chunk of another variant. This allows you to use nested chunk logic, a-la
	HyperTalk's chunk references.
 
	The class is self-cleaning.

*/

#ifndef _____VARIANT_____
#define _____VARIANT_____

#include <string>
#include <map>
#include <stack>
#include "chunk.h"

using namespace std;

class bchasm; 

class variant
{
public:
								variant				( const double& = 0 );
								variant				( const string& );
								variant				( const variant& );
								~variant			( void );

		const	double			intVal				( );
		const	string			strVal				( );

inline const	variant &		operator=			( const variant& );
inline const	variant &		operator=			( const string& );
inline const	variant &		operator=			( const double& );

inline			variant &		getMember			( const string& );
inline			void			setMember			( const string&, const string& );
inline			void			setMember			( const string&, double );
inline			void			setMember			( const string&, const variant& );

				chunk			*chunkValue;
								
inline			void			makeChunk			( const string& );
inline			void			makeChunk			( const string&, int );
inline			void			makeChunk			( const string&, int, int );

inline			void			destroyChunk		( void );

inline			void			setChunkType		( const string& );
inline			void			setChunkStart		( int );
inline			void			setChunkStop		( int );
inline			void			setChunkRange		( int, int );
inline			void			setChunkSingle		( int );

inline const	string&			getChunkType		( void );
inline const	int				getChunkStart		( void );
inline const	int				getChunkStop		( void );
				
inline			void			linkTo				( variant & );
inline			void			unLink				( void );

inline			variant&		immediateTarget		( void );
inline			variant&		finalTarget			( void );

inline			void			resolve				( void );
inline			void			flatten				( void );
				
inline const	bool			isDirty				( void );
inline const	bool			isReference			( void );
inline const	bool			isChunk				( void );

private:
		const	variant &		Copy				( const variant& );
		const	variant &		Copy				( const string& );
		const	variant &		Copy				( const double& );
		
				int				chunkStart;
				int				chunkStop;
				string			chunkType;
		
				double			longValue;
				string			stringValue;
				
				long long		myUniqueVariantID;
				
				map< string, variant>	myDataMembers;
								
				variant			*me;

				bool			vchunk;
				bool			vtype;
				bool			dirty;
				bool			reference;
				
};

inline const variant &variant::operator=( const variant &v )
{
	if ( reference )
	{
		return (*me) = v;
	}
	else
	{
		return Copy( v );
	}
}

inline const variant &variant::operator=( const string &v )
{
	if ( reference )
	{
		return (*me) = v;
	}
	else
	{
		return Copy( v );
	}
}

inline const variant &variant::operator=( const double &v )
{
	if ( reference )
	{
		return (*me) = v;
	}
	else
	{
		return Copy( v );
	}
}

inline const bool variant::isDirty( void )
{
	if ( reference )
	{
		return (*me).isDirty();
	}
	else
	{
		return dirty;
	}
}

inline const bool variant::isChunk( void )
{
	return vchunk;
}

inline const bool variant::isReference ( void )
{
	return reference;
}

inline void variant::linkTo ( variant & v )
{
	reference = true;
	me = &v;
}

inline void variant::unLink ( void )
{
	reference = false;
	me = this;
}

inline variant& variant::immediateTarget ( void )
{
	return (*me);
}

inline variant& variant::finalTarget ( void )
{
	if ( reference )
	{
		return (*me).finalTarget();
	}
	else
	{
		return (*me);
	}
}

inline void variant::resolve ( void )
{
	Copy( *me );
}

inline void variant::flatten ( void )
{
	if ( reference )
	{
		(*me).flatten();
	}
	else
	{
		Copy( *me );
	}
}

inline variant & variant::getMember ( const string& s )
{
	if ( reference )
	{
		(*me).getMember( s );
	}
	else
	{
		if ( myDataMembers.find( s ) == myDataMembers.end() )
		{
			myDataMembers.insert ( pair < string, variant > ( s, variant ( "" ) ) );
		}
		//cout << "Fetching property \"" << s << "\", which is \"" << myDataMembers[ s ].strVal() << "\"" << endl;
		return myDataMembers[ s ];
	}
}

inline void variant::setMember ( const string& s, const string& s2 )
{
	if ( reference )
	{
		(*me).setMember( s, s2 );
	}
	else
	{
		if ( myDataMembers.find( s ) != myDataMembers.end() )
		{
			myDataMembers[ s ] = s2;
		}
		else
		{
			myDataMembers.insert ( pair < string, variant > ( s, variant ( s2 ) ) );
		}
	}
}

inline void variant::setMember ( const string& s, double d )
{
	if ( reference )
	{
		(*me).setMember( s, d );
	}
	else
	{
		if ( myDataMembers.find( s ) != myDataMembers.end() )
		{
			myDataMembers[ s ] = d;
		}
		else
		{
			myDataMembers.insert ( pair < string, variant > ( s, variant ( d ) ) );
		}
	}
}

inline void variant::setMember ( const string& s, const variant& v )
{
	if ( reference )
	{
		(*me).setMember( s, v );
	}
	else
	{
		//cout << "Setting property \"" << s << "\" to \"" << v.strVal() << "\"" << endl;
		if ( myDataMembers.find( s ) != myDataMembers.end() )
		{
			myDataMembers[ s ] = v;
		}
		else
		{
			myDataMembers.insert ( pair < string, variant > ( s, v ) );
		}
	}
}

// Chunk Processing

inline void variant::makeChunk ( const string& pChunkType )
{
	vchunk = true;
	chunkType = pChunkType;
	chunkStart = 0;
	chunkStop = -1;
}

inline void variant::makeChunk ( const string& pChunkType, int pChunkLoc )
{
	vchunk = true;
	chunkType = pChunkType;
	chunkStop = chunkStart = pChunkLoc;
}

inline void variant::makeChunk ( const string& pChunkType, int pChunkStart, int pChunkStop )
{
	vchunk = true;
	chunkType = pChunkType;
	chunkStart = pChunkStart;
	chunkStop = pChunkStop;
}

inline void variant::destroyChunk ( void )
{
	vchunk = false;
	chunkType = "";
	chunkStart = 0;
	chunkStop = -1;
}

inline void variant::setChunkType ( const string& pChunkType )
{
	chunkType = pChunkType;
}

inline void variant::setChunkStart ( int pChunkStart )
{
	chunkStart = pChunkStart;
}

inline void variant::setChunkStop ( int pChunkStop )
{
	chunkStop = pChunkStop;
}

inline void variant::setChunkRange ( int pChunkStart, int pChunkStop )
{
	chunkStart = pChunkStart;
	chunkStop = pChunkStop;
}

inline void variant::setChunkSingle ( int pChunkLoc )
{
	chunkStop = chunkStart = pChunkLoc;
}

inline const string& variant::getChunkType ( void )
{
	return chunkType;
}

inline const int variant::getChunkStart ( void )
{
	return chunkStart;
}

inline const int variant::getChunkStop ( void )
{
	return chunkStop;
}

#endif
